<?php

namespace FDG\OnlineContract\Setup;

use FDG\OnlineContract\Setup\Autoupdate;

require_once ABSPATH . 'wp-admin/includes/class-wp-filesystem-base.php';

use WP_Filesystem_Base;

class Setup
{
    /**
     * Initialize a new instance of the Setup class
     *
     * @param array $settings
     */
    public function __construct($settings)
    {
        $this->settings = $settings;
        if ($this->settings['debug'] == 1) {
            $this->debug();
        }
        $this->purchaseCode = "activated";
        add_action('wp_head', [$this, 'preventIndex'], 1, 1);
    }

    /**
     * Plugin initialize, register hooks
     *
     * @return void
     */
    public function registerHooks()
    {
        register_deactivation_hook(ONLINE_CONTRACT_FILE_LOCATION, [$this, 'deactivate_plugin']);
        register_activation_hook(ONLINE_CONTRACT_FILE_LOCATION, [$this, 'activate_plugin']);
        add_action('init', [$this, 'activate_auto']);
        add_filter('plugin_action_links', [$this, 'plugin_action_links_documentation'], 10, 2);
    }

    /**
     * Setup auto update for when new version of the plugin are pushed
     *
     * @return void
     */
    function activate_auto()
    {
        $plugin_current_version = ONLINE_CONTRACT_VERSION_NUMBER;
        $plugin_remote_path = 'https://www.futuredesigngroup.com/apps/onlinecontract/update.php';
        $plugin_slug = plugin_basename(ONLINE_CONTRACT_FILE_LOCATION);
        $license_user = '';
        $license_key = $this->purchaseCode;
        new Autoupdate(
            $plugin_current_version,
            $plugin_remote_path,
            $plugin_slug,
            $license_user,
            $license_key
        );
    }

    /**
     * Functions on plugin deactivation
     *
     * @return void
     */
    function deactivate_plugin()
    {
        file_put_contents(WP_PLUGIN_DIR . '/onlinecontract/api-cache.json', '{}');
        flush_rewrite_rules();
    }

    /**
     * Functions for plugin activation, add api-cache.json if not present
     *
     * @return void
     */
    function activate_plugin()
    {
        $cache_file = ONLINE_CONTRACT_DIR . 'api-cache.json';
        if (!file_exists($cache_file)) {
            file_put_contents($cache_file, '{}');
        }
        flush_rewrite_rules();
    }

    /**
     * If debug turned on, debug php files
     *
     * @return void
     */
    function debug()
    {
        ini_set('display_errors', 1);
        ini_set('display_startup_errors', 1);
        error_reporting(E_ERROR);
    }

    /**
     * Function to prevent indexation for the contract archive and single pages
     *
     * @return void
     */
    public function preventIndex()
    {
        if (is_singular('contract')) {
            echo '<meta name="robots" content="noindex, nofollow" />' . "\r\n";
        }
    }

    /**
     * Return validation for purchase code
     *
     * @return array
     */
    public function validatePurchaseCode()
    {
        return $this->envatoApiResults();
    }

    /**
     * Create a new json file storing information for purchase
     *
     * @param string $cache_file
     * @param string $expires
     * @return void
     */
    function envatoApiResultsNew($cache_file = NULL, $expires = NULL)
    {
        global $request_type, $purge_cache, $limit_reached, $request_limit;

        if (!$cache_file) $cache_file = ONLINE_CONTRACT_DIR . 'api-cache.json';
        if (!$expires) $expires = time() - 2 * 60 * 60;

        if (!file_exists($cache_file)) {
            file_put_contents($cache_file, '{}');
        }
    }

    /**
     * Update json file storing information for purchase
     *
     * @param string $cache_file
     * @param string $expires
     * @return boolean
     */
    public function envatoApiResults()
    {
        return 1;
    }

    /**
     * Run purchase code against Envato's API
     *
     * @param string $purchaseCode
     * @return string
     */
    public function validatePurchaseCodeEnvato($purchaseCode)
    {
        $envato_url = 'https://api.envato.com/v3/market/author/sale?code=' . urlencode(trim($purchaseCode));
        $response = wp_remote_get(
            $envato_url,
            array(
                'headers' => array(
                    'Authorization' => 'Bearer xiFg3WxW6001C77aVUqUYPNlex3hmvWx'
                )
            )
        );
        if (is_array($response) && !empty($response['body'])) {
            return $response['body'];
        } else {
            return false;
        }
    }

    /**
     * Get IP address of current user
     *
     * @return string
     */
    public static function get_client_ip_server()
    {
        $ipaddress = '';
        if (isset($_SERVER['HTTP_CLIENT_IP']))
            $ipaddress = $_SERVER['HTTP_CLIENT_IP'];
        else if (isset($_SERVER['HTTP_X_FORWARDED_FOR']))
            $ipaddress = $_SERVER['HTTP_X_FORWARDED_FOR'];
        else if (isset($_SERVER['HTTP_X_FORWARDED']))
            $ipaddress = $_SERVER['HTTP_X_FORWARDED'];
        else if (isset($_SERVER['HTTP_FORWARDED_FOR']))
            $ipaddress = $_SERVER['HTTP_FORWARDED_FOR'];
        else if (isset($_SERVER['HTTP_FORWARDED']))
            $ipaddress = $_SERVER['HTTP_FORWARDED'];
        else if (isset($_SERVER['REMOTE_ADDR']))
            $ipaddress = $_SERVER['REMOTE_ADDR'];
        else
            $ipaddress = 'UNKNOWN';

        return $ipaddress;
    }

    /**
     * Add links to the plugin description
     *
     * @param array $links
     * @param string $file
     * @return array
     */
    function plugin_action_links_documentation($links, $file)
    {
        if ($file == ONLINE_CONTRACT_FILE . 'onlinecontract.php') {
            $documentation_link = '<a href="http://onlinecontract.futuredesigngroup.com/documentation" target="_blank">' . __('Documentation', 'onlinecontract') . '</a>';
            array_unshift($links, $documentation_link);
        }
        if ($this->validatePurchaseCode()) {
            if ($file == ONLINE_CONTRACT_FILE . 'onlinecontract.php') {
                $settings_link = '<a href="' . get_admin_url() . 'edit.php?post_type=contract&page=contract-settings">' . __('Settings', 'onlinecontract') . '</a>';
                array_unshift($links, $settings_link);
            }
        } else {
            if ($file == ONLINE_CONTRACT_FILE . 'onlinecontract.php') {
                $purchase_link = '<a href="' . get_admin_url() . 'edit.php?post_type=contract&page=purchase">' . __('Purchase Code', 'onlinecontract') . '</a>';
                array_unshift($links, $purchase_link);
            }
        }

        return $links;
    }
}
